/* batch_ScoreAnimations1-TD-64.jsx — 64x32 TD Animation
 * - Tints TD-Show Precomp to Secondary color
 * - Sets TeamAbbrev text content and fill to Primary color
 * - Replaces TeamLogo from Logos-Large directory
 */

(function () {
  // --- load common helpers ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) {
      try { delete $.global.GL; } catch(e) { $.global.GL = undefined; }
      $.evalFile(File(COMMON));
    }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();

  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT   = GL.env("AE_PROJECT", null);
  var CSV_PATH  = GL.env("AE_CSV", null);
  var COMP_NAME = GL.env("AE_COMP", "64x32 - TD");

  // Layer names
  var TD_SHOW_PRECOMP    = GL.cleanValue(GL.env("AE_TD_SHOW_PRECOMP", "TD-Show Precomp"));
  var TEAMABBREV_LAYER   = GL.cleanValue(GL.env("AE_TEAMABBREV_LAYER", "TeamAbbrev"));
  var LOGO_LAYER         = GL.cleanValue(GL.env("AE_LOGO_LAYER", "TeamLogo"));

  var LEAGUE    = GL.env("AE_LEAGUE", "");
  var LIMIT_STR = GL.env("AE_LIMIT", "");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = GL.env("AE_OUTDIR", "");
  var PATH_TPL  = GL.env("AE_PATH_TEMPLATE", "{league}");
  var ANIM_NAME = GL.env("AE_ANIM", "_TD_1_1");
  var RS_TPL    = GL.env("AE_RS_TEMPLATE", "Best Settings");
  var OM_TPL    = GL.env("AE_OM_TEMPLATE", "PNG Sequence");
  var PURGE     = (GL.env("AE_PURGE_BEFORE_RENDER", "1") === "1");
  var NO_RENDER = (GL.env("AE_NO_RENDER", "0") === "1");
  var QUIT_APP  = (GL.env("AE_QUIT", "1") === "1");

  var logoOpts = {
    dir: GL.env("AE_LOGO_DIR", ""),
    tpl: GL.env("AE_LOGO_PATH_TEMPLATE", "{league}/{abbr}"),
    exts: GL.env("AE_LOGO_EXTS", "png,jpg,jpeg,svg,ai,psd")
  };

  // --- start ---
  if (app.beginSuppressDialogs) { try { app.beginSuppressDialogs(); } catch(e) {} }
  app.beginUndoGroup("TD 64x32 - Batch Render");

  if (!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep = new File(PROJECT);
  if (!aep.exists) GL.fail("AE_PROJECT not found: " + PROJECT);
  try {
    var already = app.project && app.project.file && (app.project.file.fsName === aep.fsName);
    if (!already) app.open(aep);
  } catch(e) { app.open(aep); }

  if (!CSV_PATH) GL.fail("AE_CSV env not set.");
  if (!LEAGUE || GL.cleanValue(LEAGUE) === "") GL.fail("AE_LEAGUE is required.");

  var rows  = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams = GL.buildTeams(rows, LEAGUE);
  var todo  = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if (LIMIT && todo.length > LIMIT) todo = todo.slice(0, LIMIT);
  if (!todo.length) GL.fail("No teams matched league: " + LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if (!comp) GL.fail("Comp not found: " + COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  GL.rqClear();

  // --- helper functions ---

  // Tint TD-Show Precomp layer to secondary color (deep tint of all layers inside)
  function tintTDShowPrecompToSecondary(compItem, color01) {
    var layer = GL.getLayer(compItem, TD_SHOW_PRECOMP);
    if (!layer) {
      $.writeln("WARNING: TD-Show Precomp layer not found: " + TD_SHOW_PRECOMP);
      return 0;
    }
    // Use deep tint function to tint everything inside the precomp
    return GL.setTintBothColorsDeep(layer, color01, /*forceAmount100*/ true, /*depth*/ 0);
  }

  // Set TeamAbbrev text content and fill color to primary
  function setTeamAbbrevTextAndFill(compItem, abbr, color01) {
    var layer = GL.getLayer(compItem, TEAMABBREV_LAYER);
    if (!layer) {
      $.writeln("WARNING: TeamAbbrev layer not found: " + TEAMABBREV_LAYER);
      return false;
    }

    // Set text content
    GL.setTextContent(layer, abbr);

    // Set fill color (handles text layers, shape layers, and effects)
    return GL.setTextOrFillColor(layer, color01);
  }

  // --- per team loop ---
  $.writeln("\n========================================");
  $.writeln("Processing " + todo.length + " teams for league: " + LEAGUE);
  $.writeln("Comp: " + COMP_NAME);
  $.writeln("Animation: " + ANIM_NAME);
  $.writeln("========================================\n");

  for (var i = 0; i < todo.length; i++) {
    var t = todo[i];
    $.writeln("["+(i+1)+"/"+todo.length+"] Processing: " + t.abbr + " (" + t.name + ")");

    var smart = GL.computeSmartColors(GL.safeColor(t.primary), GL.safeColor(t.secondary));
    var P = smart.primary;    // Primary color
    var S = smart.secondary;  // Secondary color

    // 1) Tint TD-Show Precomp to Secondary
    var tintHits = tintTDShowPrecompToSecondary(comp, S);
    if (tintHits > 0) {
      $.writeln("  ✓ Tinted TD-Show Precomp (" + tintHits + " layers affected)");
    }

    // 2) Set TeamAbbrev text and fill to Primary
    if (setTeamAbbrevTextAndFill(comp, t.abbr, P)) {
      $.writeln("  ✓ Set TeamAbbrev: " + t.abbr + " (Primary color)");
    }

    // 3) Replace TeamLogo from Logos-Large
    var logoFile = GL.findLogoFile(t.league, t.abbr, logoOpts);
    if (logoFile) {
      $.writeln("  → Found logo: " + logoFile.fsName);

      // Import footage once
      var footage = null;
      try {
        var io = new ImportOptions(logoFile);
        if (io.canImportAs(ImportAsType.FOOTAGE)) {
          footage = app.project.importFile(io);
          $.writeln("  ✓ Imported logo: " + footage.name);
        }
      } catch(e) {
        $.writeln("  ✗ ERROR importing logo: " + e.toString());
      }

      if (footage) {
        // Check layer exists
        var logoLayer = GL.getLayer(comp, LOGO_LAYER);
        if (!logoLayer) {
          $.writeln("  ✗ Layer '" + LOGO_LAYER + "' NOT FOUND in comp");
        } else {
          // Check if it's a precomp
          if (logoLayer.source && (logoLayer.source instanceof CompItem)) {
            $.writeln("  → '" + LOGO_LAYER + "' is a precomp, looking inside...");
            var precomp = logoLayer.source;
            var replaced = 0;

            // Replace all footage layers inside the precomp
            for (var L = 1; L <= precomp.numLayers; L++) {
              var subLayer = precomp.layer(L);
              try {
                if (subLayer.source && (subLayer.source instanceof FootageItem)) {
                  subLayer.replaceSource(footage, false);
                  $.writeln("  ✓ Replaced '" + subLayer.name + "' inside precomp");
                  replaced++;
                }
              } catch(e) {
                $.writeln("  ✗ Failed to replace '" + subLayer.name + "': " + e);
              }
            }

            if (replaced === 0) {
              $.writeln("  ✗ No footage layers found inside '" + LOGO_LAYER + "' precomp");
            }
          } else if (logoLayer.source) {
            // Direct footage layer
            try {
              logoLayer.replaceSource(footage, false);
              $.writeln("  ✓ Replaced TeamLogo for " + t.abbr);
            } catch(e) {
              $.writeln("  ✗ ERROR replacing logo: " + e.toString());
            }
          } else {
            $.writeln("  ✗ Layer '" + LOGO_LAYER + "' has no source (cannot replace)");
          }
        }
      }
    } else {
      $.writeln("  ✗ Logo file not found for " + t.abbr + " at: " + logoOpts.dir + "/" + logoOpts.tpl.replace("{league}", t.league).replace("{abbr}", t.abbr));
    }

    // Pixel snap for crisp rendering
    GL.snapCompTextForPixelArt(comp);

    if (PURGE && app.purge) {
      try { app.purge(PurgeTarget.ALL_CACHES); } catch(e) {}
    }

    // Render
    if (!NO_RENDER) {
      var lc = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  → Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    // Clean up logo from project after render
    // DISABLED FOR TESTING - uncomment when working
    // var removed = GL.removeLogoByAbbr(t.abbr);
    // if (removed > 0) {
    //   $.writeln("  ✓ Removed " + removed + " logo(s) for " + t.abbr);
    // }

    $.writeln("");
  }

  $.writeln("========================================");
  $.writeln("Batch complete: " + todo.length + " teams processed");
  $.writeln("========================================");

  app.endUndoGroup();
  if (app.endSuppressDialogs) { try { app.endSuppressDialogs(); } catch(e) {} }
  if (QUIT_APP) app.quit();
})();
